<?php
/**
 * Mobile One Time Password generator/validator, based on the
 * http://motp.sourceforge.net specification.
 *
 * @package blesta
 * @subpackage blesta.components.auth.motp
 */
class Motp
{
    /**
     * @var string PIN
     */
    private $pin;
    /**
     * @var string Shared secret
     */
    private $secret;
    /**
     * @var int Length of one time password
     */
    private $length;
    /**
     * @var int Moving factor of MOTP
     */
    private $moving_factor = 10;
    /**
     * @var int The number of seconds of drift to account for (+/- 3 minutes)
     */
    private $drift = 180;

    /**
     * Creates a new Motp object with the given seeded values
     *
     * @param string $pin The pin to use for this one time password
     * @param string $secret The secret hash to use for this one time password
     * @param int $length The length of the one time password
     */
    public function __construct($pin = null, $secret = null, $length = 6)
    {
        $this->seed($pin, $secret, $length);
    }

    /**
     * Seed the Motp object with the given values
     *
     * @param string $pin The pin to use for this one time password
     * @param string $secret The secret hash to use for this one time password
     * @param int $length The length of the one time password
     */
    public function seed($pin, $secret, $length = 6)
    {
        $this->pin = $pin;
        $this->secret = $secret;
        $this->length = $length;
    }

    /**
     * Attempts to find a match for the given one time password with the given
     * time value.
     *
     * @param string $otp A suspected one time password
     * @param int $time The time around which the given otp should have originated
     * @return bool True if the one time password is valid, false otherwise
     */
    public function checkOtp($otp, $time)
    {
        // The number of iterations before and after to check for matches
        $k = (int)($this->drift / $this->moving_factor);
        // Loop through the +/- range for a match
        for ($i = -$k; $i < $k; $i++) {
            if ($otp == $this->opt(($time + ($this->moving_factor * $i)) / $this->moving_factor)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Generates a one time password based on the given time and the preset
     * secret and pin.
     *
     * @param int $time The time to use as a seed for this one time password
     * @return string The one time password generated by the given time
     * @see Motp::seed()
     */
    public function opt($time)
    {
        return $this->truncate(md5((int)$time . $this->secret . $this->pin));
    }

    /**
     * Truncates the given one time password, making it suitable for use.
     *
     * @param string $otp The one time password to truncate
     * @return string The truncated one time password
     */
    private function truncate($otp)
    {
        return substr($otp, 0, $this->length);
    }
}
